
; Duct Heat Transfer Controller


	ERRORLEVEL -302
	ERRORLEVEL -306

	list P=16F1459
	#include p16f1459.inc

;Program Configuration Register 1
		__CONFIG    _CONFIG1, _FOSC_INTOSC & _WDTE_ON & _PWRTE_ON & _MCLRE_ON & _CP_OFF & _BOREN_OFF & _CLKOUTEN_OFF &_IESO_OFF & _FCMEN_OFF

;Program Configuration Register 2
		__CONFIG    _CONFIG2, _WRT_OFF & _CPUDIV_NOCLKDIV & _USBLSCLK_24MHz & _PLLMULT_4x & _PLLEN_DISABLED & _STVREN_ON & _BORV_HI & _LPBOR_OFF & _LVP_OFF 

;..................................................................................................................................
; define;
; Bank 0 RAM 
; Stored data
TEMP_VAL		equ	H'20'		; Delay temporary Value
STORE1			equ	H'21'		; delay counter
STORE2			equ	H'22'		; delay counter
TEMP			equ	H'23'		; temporary register for WAIT routine
PRESENCE1		equ H'24'		; presence flag for sensor 1
PRESENCE2		equ H'25'		; presence flag for sensor 2
DATA_BYTE		equ	H'26'		; 8-bit byte to or from temperature sensor
COUNT			equ	H'27'		; counter for read/write data
MIN_IT			equ	H'28'		; minute timer
MINUTE			equ	H'29'		; minute time period flag
RATE			equ	H'2A'		; Flash write rate
FLASH			equ	H'2B'		; LED flash flag (flash if bit 0 is set)
TIMERms			equ	H'2C'		; count down timer most significant byte
TIMERls			equ	H'2D'		; count down timer least significant byte
STOREI1			equ	H'2E'		; interrupt delay counter
STOREI2			equ	H'2F'		; interrupt delay counter	
T_MULTms		equ	H'30'		; timer multiplier for 15minutes D'1717' (0.524288 x 1717 = 15m)
T_MULTls		equ	H'31'		; timer multiplier ls byte
AUTO_STATUS		equ	H'32'		; automatic operation status (on or off)
TEMP1ms			equ	H'33'		; sensor 1 temperature ms value
TEMP1ls			equ	H'34'		; sensor 1 temperature ls value
TEMP2ms			equ	H'35'		; sensor 2 temperature ms value
TEMP2ls			equ	H'36'		; sensor 2 temperature ls value
BCD1			equ	H'37'		; setting for BCD switch BCD1
BCD2			equ	H'38'		; setting for BCD switch BCD2
BCD3			equ	H'39'		; setting for BCD switch BCD3
BCD4			equ	H'3A'		; setting for BCD switch BCD4
SENSOR_OUT		equ	H'3B'		; sensor disconnected flag
MODE			equ	H'3C'		; operating mode
OVER			equ	H'3D'		; temperature over flag
KEEP			equ	H'3E'		; temperature calculations /comparisons keep byte
CALCms			equ	H'3F'		; comparison calculations temporary value ms byte
CALCls			equ	H'40'		; comparison calculations temporary value ls byte
TEMP1ms_OLD		equ	H'41'		; old temperature value for fire alarm detection ms byte
TEMP1ls_OLD		equ	H'42'		; old temperature value for fire alarm detection ls byte
TEMP2ms_OLD		equ	H'43'		; old temperature value for fire alarm detection ms byte
TEMP2ls_OLD		equ	H'44'		; old temperature value for fire alarm detection ls byte
ROTATE			equ	H'45'		; used in testing of timer 
BCD4_LAST		equ	H'46'		; MODE setting for with BCD4. Check previous/last value detecting a change in setting
RUN_CHECK		equ	H'47'

; all banks
FIRE			equ	H'70'		; fire flag
SEC_1			equ	H'71'		; 1 second counter (0.524288s x 2=1.048576s)
PCON_STORE		equ	H'72'		; store PCON to check power on reset/watchdog timeout flag
TEST_TIME		equ	H'73'		; timer

; A/D test
TEST			equ	H'79'		; ms byte of A/D for temperature testing via A/D
; used for testing temperature readings shown on Flash
LOOPS			equ	H'7A'		; flash write loop counter
INTCON_STORE	equ	H'7B'		; flash write check GIE setting
READADDMS		equ	H'7C'		; ms byte flash
READADDLS		equ	H'7D'		; ls byte flash

; test bytes for showing temperature on PWM2 (RC6) output
TESTms			equ	H'7E'		; ms byte of test
TESTls			equ	H'7F'		; ls byte of test	

		
;.................................................................

	org		0
	goto	MAIN
	org     4				; interrupt vector @ 4
	goto	INTERRUPT

; Lookup tables for non linear value steps

SETTING1 ; BCD1 delta T. 0.5 degrees per value
	addwf	PCL,f
	retlw	D'2'			; BCD=0, value=1 degree
	retlw	D'3'			; BCD=1, value=1.5 degrees
	retlw	D'4'			; BCD=2, value=2 degrees
	retlw	D'6'			; BCD=3, value=3 degrees
	retlw	D'8'			; BCD=4, value=4 degrees
	retlw	D'10'			; BCD=5, value=5 degrees
	retlw	D'12'			; BCD=6, value=6 degrees
	retlw	D'16'			; BCD=7, value=8 degrees
	retlw	D'20'			; BCD=8, value=10 degrees
	retlw	D'22'			; BCD=9, value=11 degrees
	retlw	D'22'

SETTING2 ; BCD2 Hysteresis. 0.5 degrees per value
	addwf	PCL,f
	retlw	D'1'			; BCD=0, value=0.5 degrees
	retlw	D'2'			; BCD=1, value=1 degree
	retlw	D'3'			; BCD=2, value=1.5 degrees
	retlw	D'4'			; BCD=3, value=2 degrees
	retlw	D'6'			; BCD=4, value=3 degrees
	retlw	D'8'			; BCD=5, value=4 degrees
	retlw	D'10'			; BCD=6, value=5 degrees
	retlw	D'12'			; BCD=7, value=6 degrees
	retlw	D'16'			; BCD=8, value=8 degrees
	retlw	D'20'			; BCD=9, value=10 degrees
	retlw	D'20'

SETTING3 ; BCD3 timer 15m per value
	addwf	PCL,f
	retlw	D'1'			; BCD=0, value=15m
	retlw	D'2'			; BCD=1, value=30m
	retlw	D'4'			; BCD=2, value=1h
	retlw	D'8'			; BCD=3, value=2h
	retlw	D'12'			; BCD=4, value=3h
	retlw	D'16'			; BCD=5, value=4h 
	retlw	D'20'			; BCD=6, value=5h 
	retlw	D'24'			; BCD=7, value=6h
	retlw	D'32'			; BCD=8, value=8h 
	retlw	D'48'			; BCD=9, value=12h
	retlw	D'48'

INTERRUPT

; check interrupt source
	movlb	D'0'			; bank 0
	btfss	PIR1,TMR1IF		; timer1 interrupt flag occurs each 524.288ms
	retfie
	bcf		PIR1,TMR1IF		; clear flag

; 1 minute timer
; 114 interrupts x 524.288ms = 60 seconds 

	decfsz	MIN_IT,f
	goto	FIFTEEN_MIN
; if fire alarm is off, bypass
	btfsc	PORTB,7
	goto	FIFTEEN_MIN
	movlw	D'114'			; D'114' reload timer for 1 minute
	movwf	MIN_IT
	bsf		MINUTE,0		; set minute flag		
	bsf		MINUTE,1

FIFTEEN_MIN

; 15 minute timer
; bypass if not mode 1 (timer operation)
	movf	MODE,w
	xorlw	D'1'			; timer mode
	btfss	STATUS,Z
	goto	CK_FLASH

	movf	T_MULTms,w		; 15m per value count down, if zero bypass ms byte decrement
	btfss	STATUS,Z		; if ms byte is zero, check ls byte
	goto	DEC_TIMER
	movf	T_MULTls,w		; ls byte
	btfsc	STATUS,Z
	goto	RELOAD_T		; if both zero, load 15m value
; ls=>0
	decf	T_MULTls,f		; ls decreased
	goto	CK_FLASH

DEC_TIMER ; ms >0, ls=0

	movf	T_MULTls,w
	btfss	STATUS,Z
; decrement ms byte if not 0
	goto	MS_RED
	decf	T_MULTls,f
	decf	T_MULTms,f

	goto	CK_FLASH
MS_RED ; ls not 0
	decf	T_MULTls,f	
	goto	CK_FLASH
RELOAD_T
	movlw	H'6'			; D1716 for 15m (H'6B4') 524.288ms x 1716= 15m
	movwf	T_MULTms		; timer multiplier for 15m per value for timerms/timerls
	movlw	H'B4'
	movwf	T_MULTls	

	movf	TIMERms,w		; 15m per value count down, if zero bypass ms byte decrement
	btfss	STATUS,Z		; if ms byte is zero, check ls byte
	goto	DEC_TIM
	movf	TIMERls,w		; ls byte
	btfsc	STATUS,Z
	goto	END_TIMER		; if both zero, bypass decrement
	decfsz	TIMERls,f
	goto	CK_FLASH
	goto	END_TIMER
DEC_TIM
	movf	TIMERls,w
	btfss	STATUS,Z
; decrement ms byte if not 0
	goto	MS_REDU
	decf	TIMERls,f
	decf	TIMERms,f

	goto	CK_FLASH
MS_REDU	; ls is zero
	decf	TIMERls,f

; if both zero, end of timer
	movf	TIMERms,w		; if zero 
	btfss	STATUS,Z		; if ms byte is zero, check ls byte
	goto	CK_FLASH
	movf	TIMERls,w		; ls byte
	btfss	STATUS,Z
	goto	CK_FLASH
END_TIMER
; relay1 off and LED off
	clrf	FLASH
	movlb	D'2'
	bcf		LATB,6			; LED off
	bcf		LATC,3			; relay1 off
	movlb	D'0'	

CK_FLASH

	btfss	FLASH,0			; if set, then flash LED
	goto	BY_FLASH
; Momentary flash on
	incf	SEC_1,f	
	btfss	SEC_1,2
	retfie
	clrf	SEC_1
	movlb	D'2'			; bank 2
	bcf		LATB,6			; LED off
	movlb	D'0'			; bank 0
	movlw	D'186'			; 100ms
	call	DELAYI			; interrupt delay routine
	movlb	D'2'			; bank 2
	bsf		LATB,6			; LED on
	movlb	D'0'			; bank 0
	retfie
BY_FLASH
	btfss	FLASH,1			; if set then momentary flash	
	retfie
; if SEC_1 bit 2 is set, then flash momentarily
; Momentary flash off
	incf	SEC_1,f	
	btfss	SEC_1,2
	retfie
	clrf	SEC_1
	movlb	D'2'			; bank 2
	bsf		LATB,6			; LED on
	movlb	D'0'			; bank 0
	movlw	D'93'			; 50ms
	call	DELAYI			; interrupt delay routine
	movlb	D'2'			; bank 2
	bcf		LATB,6			; LED off
	movlb	D'0'			; bank 0
	retfie

;------------------------------------------------------------------------------------------------------

MAIN

; store power control, check if watchdog timeout
	movlb	D'1'			; bank 1
	movf	PCON,w
	movlb	D'0'			; bank 0
	movwf	PCON_STORE		; store PCON reset flags
	btfsc	PCON_STORE,4	; if not wdt timeout then begin at setup
	goto	SETUP
	clrwdt
	movlb	D'1'			; bank 1
	bsf		PCON,4			; set wdt over flag
	movlb	D'0'			; bank 0

SETUP;

; set inputs/outputs
; if wdt timeout bypass clearing ports
	btfss	PCON_STORE,4	; if wdt timeout bypass clearing
	goto	USB
	movlb	D'2'			; latch, bank 2
	clrf	LATA
	clrf	LATB
	clrf	LATC
USB
	movlb	D'29'
	bcf		UCON,3		; USB off
	bcf		UCON,1
; weak pullups off/on
	movlb	D'4'		; bank4	WPUA/B
	clrf	WPUA
	movlw	B'10000000' ; RB7
	movwf	WPUB
; set I/O
	movlb	D'1'		; bank1	TRISA/B/C
	movlw   B'00001011'	; I/O 
	movwf   TRISA		; port A data direction register
	movlw	B'10110000'	; I/O 
	movwf	TRISB		; port B data direction register
	movlw	B'00110100'
	movwf	TRISC		; port C data direction 
; options
	movlw	B'00000111'	; weak pullups set via WPUA/B TMR0/256
	movwf	OPTION_REG	; 
; oscillator	
	movlw	B'00110110'	; for 4MHz;
	movwf	OSCCON		; osc
; watchdog timer
	movlw	B'00011011'	; 8s watchdog
	movwf	WDTCON
; analog inputs. None
	movlb	D'3'		; bank3	
	movlw	B'00000000'	; 
	movwf	ANSELA
	movwf	ANSELB
	movwf	ANSELC

	movlb	D'0'		; bank 0
; timer1
	movlw	B'00110001'	; B'00110001' timer1 /8  prescaler for 524.28ms
	movwf	T1CON

;******************************************************************

; test for PWM to show temperature reading (or other readings) at RC6. ie PWM DC voltage corresponds to temperature.
; 0 degrees C is half supply. PIC supply voltage x 256/512. 0.5 degrees per lsb of PWM using 512 bits resolution at around 10mV/0.5 deg C 
; Disconnect 1k resistor at base of Q2 for this test
; set up PWM2 at RC6
;	movlw	D'255'
;	movwf	PR2
;	movlw	B'00000110' ; /64
;	movwf	T2CON
;	movlw	B'11100000'
;	movlb	D'12'		; bank 12
;	movwf	PWM2CON
; initial data
;	movlw	D'0'		; for temperature half way voltage D'127'
;	movwf	PWM2DCH
;	movlw	D'0'		; for temperature use B'01000000'
;	movwf	PWM2DCL
;	movlb	D'0'		; bank 0
; End Test

;************************************************************************
; if wdt timeout bypass initial values
	btfss	PCON_STORE,4	; if not wdt timeout then initialise 
	goto	LOOP

; short start up delay
	call	DELAYxms	; 300ms
; initial values
INIT; initialise
	
; if not a software reset, load MODE with FF. If software reset load MODE with BCD4 setting
	movlw	H'FF'
	btfsc	PCON_STORE,2	; POR stored value. if a software reset, BCD4 changed from last time has occurred
	movwf	BCD4_LAST	; load the FF indicating a change occurred	
	clrf	FLASH		; LED flash flag
	clrf	TIMERms		; timer on operation, most significant byte, BCD3 setting dependent, if ms and ls are zero, no timeout
	clrf	TIMERls		; timer on operation, least significant byte
	clrf	AUTO_STATUS	; automatic operation status (on or off)
	bsf		AUTO_STATUS,0	; automatic on at power up
	clrf	FIRE		; fire flag
	clrf	MINUTE		; fire alarm minute flag
	clrf	TEMP1ms_OLD	; old temperature1 value for fire alarm detection ms byte
	clrf	TEMP1ls_OLD
	clrf	TEMP2ms_OLD	; old temperature2 value for fire alarm detection ms byte
	clrf	TEMP2ls_OLD
	clrf	OVER		; over temperature flags
	clrf	SENSOR_OUT	; sensor disconnected flag
	clrf	T_MULTms	; timer multiplier
	clrf	T_MULTls

; setup interrupts

	movlb	D'0'		; bank 0
	bsf		INTCON,PEIE	; enable peripheral interrupts

	movlb	D'1'		; bank 1
	bsf		PIE1,TMR1IE	; enable timer1 interrupt
	movlb	D'0'		; bank 0
	bcf		PIR1,TMR1IF	; timer1 interrupt flag
	bsf		INTCON,GIE	; global interrupts enable 

LOOP; program cycles back to here

	clrwdt				; watchdog 8s timeout

; if Fire alarm, bypass to TEMP_LOOP
	btfsc	FIRE,0		; if set do temperature loop only
	goto	TEMP_LOOP
		
; read BCD switches
; BCD1 temperature difference
	movlb	D'2'		; bank 2
	bsf	 	LATA,4		; enable read of BCD1
	movlb	D'0'		; bank 0
; RA1, 1; RC5, 2; RA0, 4 and RC4, 8
	nop
	clrf	BCD1
	btfsc	PORTA,1		; '1'if set, set bit 0
	bsf		BCD1,0
	btfsc	PORTC,5		; '2'if set, set bit 1
	bsf		BCD1,1
	btfsc	PORTA,0		; '4'if set, set bit 2
	bsf		BCD1,2
	btfsc	PORTC,4		; '8'if set, set bit 3	 
	bsf		BCD1,3

	movlb	D'2'		; bank 2
	bcf	 	LATA,4		; BCD1 off
	movlb	D'0'		; bank 0

	movf	BCD1,w		; BCD setting
	call	SETTING1	; lookup table for non linear values
	movwf	BCD1		; modified values

	nop
	nop
	nop

; BCD2 hysteresis
	movlb	D'2'
	bsf	 	LATA,5		; enable read of BCD2
	movlb	D'0'

; RA1, 1; RC5, 2; RA0, 4 and RC4, 8
	nop
	clrf	BCD2
	btfsc	PORTA,1		; '1'if set, set bit 0
	bsf		BCD2,0
	btfsc	PORTC,5		; '2'if set, set bit 1
	bsf		BCD2,1
	btfsc	PORTA,0		; '4'if set, set bit 2
	bsf		BCD2,2
	btfsc	PORTC,4		; '8'if set, set bit 3
	bsf		BCD2,3

	movlb	D'2'
	bcf	 	LATA,5		; BCD2 off
	movlb	D'0'
	movf	BCD2,w		; BCD setting

	call	SETTING2	; lookup table for non linear values
	movwf	BCD2		; modified values
	nop
	nop
	nop

; BCD3 timer steps
	movlb	D'2'
	bsf	 	LATC,0		; enable read of BCD3
	movlb	D'0'

; RA1, 1; RC5, 2; RA0, 4 and RC4, 8
	nop
	clrf	BCD3
	btfsc	PORTA,1		; '1'if set, set bit 0
	bsf		BCD3,0
	btfsc	PORTC,5		; '2'if set, set bit 1
	bsf		BCD3,1
	btfsc	PORTA,0		; '4'if set, set bit 2
	bsf		BCD3,2
	btfsc	PORTC,4		; '8'if set, set bit 3
	bsf		BCD3,3
	movlb	D'2'
	bcf	 	LATC,0		; BCD3 off
	movlb	D'0'
	
	movf	BCD3,w		; BCD setting
	call	SETTING3	; lookup table for non linear values
	movwf	BCD3		; modified values

	nop
	nop
	nop
 
; BCD4 Mode
	movlb	D'2'
	bsf	 	LATC,1		; enable read of BCD4
	movlb	D'0'
; RA1, 1; RC5, 2; RA0, 4 and RC4, 8
	nop
	clrf	BCD4
	btfsc	PORTA,1		; '1'if set, set bit 0
	bsf		BCD4,0
	btfsc	PORTC,5		; '2'if set, set bit 1
	bsf		BCD4,1
	btfsc	PORTA,0		; '4'if set, set bit 2
	bsf		BCD4,2
	btfsc	PORTC,4		; '8'if set, set bit 3
	bsf		BCD4,3
	movlb	D'2'
	bcf	 	LATC,1		; BCD4 off
	movlb	D'0'


	movf	BCD4,w
	xorwf	BCD4_LAST,w
	btfsc	STATUS,Z	; if BCD4 and BCD4_LAST are the same then BCD4 hasn't changed
	goto	MODE_CHECK
	movf	BCD4,w
	movwf	BCD4_LAST
	reset				; clear registers for fresh beginning of operation with new MODE setting
;
; Check Mode
MODE_CHECK	

;*********************************************************
; testing breadboard mockup that does not have BCD switches except BCD4
; make sure these are deleted or remarked out in final version
; settings via BCD switches in software testing
;	movlw	D'6'		; delta T at 6 deg C
;	movwf	BCD1
;	movlw	D'1'		; hysteresis 0.5 x value deg C
;	movwf	BCD2
;	movlw	D'2'		; Timer 30m
;	movwf	BCD3

; show timer or other value via pwm. Note: remark out pwm initialisation also
;	movf	TIMERls,w		; or BCD3,w; value to show
;	movwf	ROTATE
;	lslf	ROTATE,f
;	rlf		ROTATE,f
;	rlf		ROTATE,f
;	rlf		ROTATE,f
;	rlf		ROTATE,w
;	movlb	D'12'
;	movwf	PWM2DCH
;	clrf	PWM2DCL
;	movlb	D'0'

; ended	
;*********************************************************

	movf	BCD4,w		; mode setting
	btfsc	STATUS,Z	; if zero then MODE0 (manual operation)
	goto	MODE0
	movf	BCD4,w
	xorlw	D'1'
	btfsc	STATUS,Z
	goto	MODE1		; Manual fan timed operation
	movf	BCD4,w
	xorlw	D'2'
	btfsc	STATUS,Z
	goto	MODE2		; Automatic
	goto	MODE2		; any setting over 2 is automatic

MODE0
; Manual operation
; S1 short press, fan on
; S1 long press, fan off
	clrf	MODE		; mode=0

; check switch
	btfsc	PORTB,5		
	goto	TEMP_LOOP		; switch not pressed

	bsf		PORTC,7			; apply short beep sound
	call	DELAYms			; 100ms
	bcf		PORTC,7			; beep off
; check if kept closed for 500ms
	call	DELAYms			; 100ms 
	btfsc	PORTB,5			; if open
	goto	OPEN_0			; open for mode0
	call	DELAYms			; 100ms 
	btfsc	PORTB,5			; if open
	goto	OPEN_0			; open for mode0
	call	DELAYms			; 100ms 
	btfsc	PORTB,5			; if open
	goto	OPEN_0			; open for mode0
	call	DELAYms			; 100ms 
	btfsc	PORTB,5			; if open
	goto	OPEN_0			; open for mode0
	call	DELAYms			; 100ms 
; check switch
; if still closed after 500ms, switch off 
	btfsc	PORTB,5			; if closed
	goto	OPEN_0			; open for mode0
	movlb	D'2'
	bcf		LATB,6			; LED off
	bcf		LATC,3			; relay off
	movlb	D'0'
	clrf	OVER			; temperature over flags
; clear timer
	clrf	TIMERls
	clrf	TIMERms
; apply double beep
	bsf		PORTC,7			; apply short beep sound
	call	DELAYms			; 100ms
	bcf		PORTC,7			; beep off
	call	DELAYms
	bsf		PORTC,7			; apply short beep sound
	call	DELAYms			; 100ms
	bcf		PORTC,7			; beep off

; reset flash
	clrf	FLASH
	movlb	D'2'			; bank for latch
	bcf		LATB,6			; LED off
	movlb	D'0'	
	
; wait for switch open
WAIT_S1_MODE0
	btfsc	PORTB,5
	goto	OPE_MODE0
	call	DELAYms			; wait 100ms
	goto	WAIT_S1_MODE0
OPE_MODE0
	call	DELAYms
	btfss	PORTB,5			; if still open, end loop
	goto	WAIT_S1_MODE0
	goto	TEMP_LOOP
OPEN_0
; switch on LED and fan	
	movlb	D'2'
	bsf		LATB,6			; LED on
	bsf		LATC,3			; relay on
	movlb	D'0'
	goto	TEMP_LOOP

MODE1
; manual timed operation
	movlw	D'1'
	movwf	MODE
; timer and manual operation
; S1 short press, fan on. Number of short presses sets time.
; S1 long press, fan off

; check switch
	btfsc	PORTB,5		
	goto	TEMP_LOOP		; switch not pressed

	bsf		PORTC,7			; apply short beep sound
	call	DELAYms			; 100ms
	bcf		PORTC,7			; beep off

; check if kept closed for 500ms
	call	DELAYms			; 100ms 
	btfsc	PORTB,5			; if open
	goto	OPEN_1			; open for mode1
	call	DELAYms			; 100ms 
	btfsc	PORTB,5			; if open
	goto	OPEN_1			; open for mode1
	call	DELAYms			; 100ms 
	btfsc	PORTB,5			; if open
	goto	OPEN_1			; open for mode1
	call	DELAYms			; 100ms 
	btfsc	PORTB,5			; if open
	goto	OPEN_1			; open for mode1
	call	DELAYms			; 100ms 

; check switch
; if still closed switch off 
	btfsc	PORTB,5			; if closed
	goto	OPEN_1			; open for mode1
	movlb	D'2'
	bcf		LATB,6			; LED off
	bcf		LATC,3			; relay off
	movlb	D'0'	
	clrf	OVER			; temperature over flags
; clear timer
	clrf	TIMERls
	clrf	TIMERms

; apply double beep
	bsf		PORTC,7			; apply short beep sound
	call	DELAYms			; 100ms
	bcf		PORTC,7			; beep off
	call	DELAYms
	bsf		PORTC,7			; apply short beep sound
	call	DELAYms			; 100ms
	bcf		PORTC,7			; beep off

; reset flash
	clrf	FLASH
	movlb	D'2'			; bank for latch
	bcf		LATB,6			; LED off
	movlb	D'0'
	
; wait for switch open
WAIT_S1_MODE1
	btfsc	PORTB,5
	goto	OPE_MODE1
	call	DELAYms			; wait 100ms
	goto	WAIT_S1_MODE1
OPE_MODE1
	call	DELAYms
	btfss	PORTB,5			; if still open, end loop
	goto	WAIT_S1_MODE1
	goto	TEMP_LOOP
OPEN_1
; switch on LED and fan	
	movlb	D'2'
	bsf		LATB,6			; LED on
	bsf		LATC,3			; relay on
	movlb	D'0'
; restart timer multiplier
	bcf		INTCON,GIE
	movlw	H'6'			; D1716 for 15m (H'6B4')
	movwf	T_MULTms		; timer multiplier for 15m per value for timerms/timerls
	movlw	H'B4'
	movwf	T_MULTls
	bsf		INTCON,GIE

; set timer 16 bit increased by BCD3 value on each 'on' press
; BCD3 switch has timer steps (15minutes per value on BCD3 switch)
; 
	movf	BCD3,w
	addwf	TIMERls,f
	btfsc	STATUS,C		; if carry, increase ms byte
	incf	TIMERms,f
	goto	TEMP_LOOP
	

MODE2
; automatic
	movlw	D'2'
	movwf	MODE
; S1 short press, auto operation is on. 
; S1 long press, auto is off

; check switch
	btfsc	PORTB,5		
	goto	TEMP_LOOP		; switch not pressed

	bsf		PORTC,7			; apply short beep sound
	call	DELAYms			; 100ms
	bcf		PORTC,7			; beep off

; check if kept closed for 500ms
	call	DELAYms			; 100ms 
	btfsc	PORTB,5			; if open
	goto	OPEN_2			; open for mode2
	call	DELAYms			; 100ms 
	btfsc	PORTB,5			; if open
	goto	OPEN_2			; open for mode2
	call	DELAYms			; 100ms 
	btfsc	PORTB,5			; if open
	goto	OPEN_2			; open for mode2
	call	DELAYms			; 100ms 
	btfsc	PORTB,5			; if open
	goto	OPEN_2			; open for mode2
	call	DELAYms			; 100ms 

; check switch
; if still closed switch off 
	btfsc	PORTB,5			; if closed
	goto	OPEN_2			; open for mode2
	movlb	D'2'
	bcf		LATB,6			; LED off
	bcf		LATC,3			; relay off
	movlb	D'0'
	clrf	OVER			; temperature over flags
; reset flash
	bsf		FLASH,1			; momentary flash on
	bcf		FLASH,0			; momentary flash off
; clear timer
	clrf	TIMERls
	clrf	TIMERms
	clrf	AUTO_STATUS		; automatic mode status (off)

; apply double beep
	bsf		PORTC,7			; apply short beep sound
	call	DELAYms			; 100ms
	bcf		PORTC,7			; beep off
	call	DELAYms
	bsf		PORTC,7			; apply short beep sound
	call	DELAYms			; 100ms
	bcf		PORTC,7			; beep off
	
; wait for switch open
WAIT_S1_MODE2
	btfsc	PORTB,5
	goto	OPE_MODE2
	call	DELAYms			; wait 100ms
	goto	WAIT_S1_MODE2
OPE_MODE2
	call	DELAYms
	btfss	PORTB,5			; if still open, end loop
	goto	WAIT_S1_MODE2
	goto	TEMP_LOOP
OPEN_2
	bsf		AUTO_STATUS,0	; automatic mode status (ON)
; switch on LED and fan	if delta T over
	movf	OVER,w
	btfsc	STATUS,Z
	goto	TEMP_LOOP
	movlb	D'2'
	bsf		LATB,6			; LED on
	bsf		LATC,3			; relay on
	movlb	D'0'

;...................................................................................
TEMP_LOOP

; read temperature
; check switch S1 between temperature read sections. If closed goto LOOP
; sensor 1


RST1
	call	RESET1
	btfsc	PRESENCE1,0		; if set then sensor1 connected
	goto	CONNECTED1
	btfsc	SENSOR_OUT,1	; if set then sensor is already detected as out
	goto	TEMP_VAL1
; if not connected, buzzer sounds once 	
; apply beep

RST11
	bsf		SENSOR_OUT,1	; sensor1 disconnected flag

	bsf		PORTC,7			; apply short beep sound
	call	DELAYms			; 100ms
	bcf		PORTC,7			; beep off
	movlw	D'10'			; 1 second
	call	XMS				; timer
	call	RESET1
	btfsc	PRESENCE1,0		; if set then sensor1 connected
	goto	INIT			; initialise program
	goto	RST2

CONNECTED1
	bcf		SENSOR_OUT,1	; sensor connected
	btfss	PORTB,5			; check S1
	goto	LOOP	

	movlw	H'CC'			; skip ROM
	call	XMIT1			; send data

	btfss	PORTB,5			; check S1
	goto	LOOP

	movlw	H'44'			; do temperature conversion
	call	XMIT1

	btfss	PORTB,5			; check S1
	goto	LOOP

	movlw	D'5'
	movwf	TEST_TIME		; test timer counter (5=500ms when 1 used with XMS delay)
READY1
	movlw	D'1'			; 1 bit ready 
	movwf	COUNT
	clrf	DATA_BYTE		; start with H00
	call	RCV_DATA1

	btfss	PORTB,5			; check S1
	goto	LOOP

; *** if DATA_BYTE,7 bit not received as set within time; bypass

	btfsc	DATA_BYTE,7		; if set then data ready
	goto	RESET_WAIT1
	decfsz	TEST_TIME,f
	goto	TEST_T1
	goto	RST2			; check sensor 2
TEST_T1
	movlw	D'1'			; 100ms (total of 500ms with TEST_TIME counter at 5)
	call	XMS				; timer
	goto	READY1			; not available yet
	
; data ready
RESET_WAIT1	
	call	RESET1
	btfsc	PRESENCE1,0		; if set then sensor1 connected
	goto	CONNECT1
	btfsc	SENSOR_OUT,1	; if set then sensor is already detected as out
	goto	TEMP_VAL1
; if not connected, buzzer sounds once each 2s	
; apply beep

	bsf		SENSOR_OUT,1	; sensor1 disconnected flag
	bsf		PORTC,7			; apply short beep sound
	call	DELAYms			; 100ms
	bcf		PORTC,7			; beep off
	movlw	D'10'			; 1 second
	call	XMS				; timer
	goto	RST2			; 
CONNECT1
	bcf		SENSOR_OUT,1	; sensor connected
	btfss	PORTB,5			; check S1
	goto	LOOP	

	movlw	H'CC'			; skip ROM
	call	XMIT1			; send data

	btfss	PORTB,5			; check S1
	goto	LOOP

	movlw	H'BE'			; read temperature
	call	XMIT1

	btfss	PORTB,5			; check S1
	goto	LOOP
; ls byte read
	clrf	DATA_BYTE		; start with H00
	call	RCV1			; read ls byte
	movf	DATA_BYTE,w
	movwf	TEMP1ls
	
	btfss	PORTB,5			; check S1
	goto	LOOP

; ms byte read
	clrf	DATA_BYTE		; start with H00
	call	RCV1			; read ms byte
	movf	DATA_BYTE,w
	movwf	TEMP1ms

	btfss	PORTB,5			; check S1
	goto	LOOP

; sensor 2

RST2
	call	RESET2
	btfsc	PRESENCE2,0		; if set then sensor1 connected
	goto	CONNECTED2

	btfsc	SENSOR_OUT,2	; if set then sensor is already detected as out
	goto	TEMP_VAL1
; if not connected, buzzer sounds twice 
; apply beep
RST22
	bsf		SENSOR_OUT,2	; sensor2 disconnected flag
	bsf		PORTC,7			; apply short beep sound
	call	DELAYms			; 100ms
	bcf		PORTC,7			; beep off
	call	DELAYms			; 100ms
	bsf		PORTC,7			; apply short beep sound
	call	DELAYms			; 100ms
	bcf		PORTC,7			; beep off
	movlw	D'10'			; 1 second
	call	XMS				; timer
	call	RESET2
	btfsc	PRESENCE2,0		; if set then sensor2 connected
	goto	INIT			; once connected reinitialise program
	goto	TEMP_VAL1			

CONNECTED2
	bcf		SENSOR_OUT,2	; sensor in
	btfss	PORTB,5			; check S1
	goto	LOOP	

	movlw	H'CC'			; skip ROM
	call	XMIT2			; send data

	btfss	PORTB,5			; check S1
	goto	LOOP

	movlw	H'44'			; do temperature conversion
	call	XMIT2

	btfss	PORTB,5			; check S1
	goto	LOOP

	movlw	D'5'
	movwf	TEST_TIME		; test timer counter (5=500ms when 1 used with XMS delay)
READY2
	movlw	D'1'			; 1 bit read for data ready 
	movwf	COUNT
	clrf	DATA_BYTE		; start with H00
	call	RCV_DATA2

	btfss	PORTB,5			; check S1
	goto	LOOP

	btfss	DATA_BYTE,7		; if set then data ready
	goto	READY2			; not available yet

; *** if DATA_BYTE,7 bit not received as set within time; bypass

	btfsc	DATA_BYTE,7		; if set then data ready
	goto	RESET_WAIT2
	decfsz	TEST_TIME,f
	goto	TEST_T2
	goto	COMP			; check comparison
TEST_T2
	movlw	D'1'			; 100ms
	call	XMS				; timer
	goto	READY2			; not available yet

; data ready
RESET_WAIT2
	call	RESET2
	btfsc	PRESENCE2,0		; if set then sensor2 connected
	goto	CONNECT2
	btfsc	SENSOR_OUT,2	; if set then sensor is already detected as out
	goto	TEMP_VAL1
; if not connected, buzzer sounds once each 2s	
; apply beep
	bsf		SENSOR_OUT,2	; sensor2 disconnected flag
	bsf		PORTC,7			; apply short beep sound
	call	DELAYms			; 100ms
	bcf		PORTC,7			; beep off
	call	DELAYms			; 100ms
	bsf		PORTC,7			; apply short beep sound
	call	DELAYms			; 100ms
	bcf		PORTC,7			; beep off
	movlw	D'10'			; 1 second
	call	XMS				; timer
	goto	TEMP_VAL1		; 
CONNECT2
	bcf		SENSOR_OUT,2	; sensor connected
	btfss	PORTB,5			; check S1
	goto	LOOP	

	movlw	H'CC'			; skip ROM
	call	XMIT2			; send data

	btfss	PORTB,5			; check S1
	goto	LOOP

	movlw	H'BE'			; read temperature
	call	XMIT2

	btfss	PORTB,5			; check S1
	goto	LOOP
; ls byte read
	clrf	DATA_BYTE		; start with 00
	call	RCV2			; read ls byte
	movf	DATA_BYTE,w
	movwf	TEMP2ls
	
	btfss	PORTB,5			; check S1
	goto	LOOP

; ms byte read
	clrf	DATA_BYTE		; start with 00
	call	RCV2			; read ms byte
	movf	DATA_BYTE,w
	movwf	TEMP2ms

	btfss	PORTB,5			; check S1
	goto	LOOP
	goto	COMP

; temperature values
TEMP_VAL1
; if a sensor is disconnected set temperatures at 0
	movf	SENSOR_OUT,w	; sensor disconnected flags at bit1 and bit2
	btfsc	STATUS,Z
	goto	COMP			; both connected
	clrf	TEMP2ls			; clear temperature readings to 0 for both sensors
	clrf	TEMP1ls
	clrf	TEMP2ms
	clrf	TEMP1ms

	
COMP
	
; do temperature comparisons
; TEMP1ms		; sensor 1 temperature ms value
; TEMP1ls		; sensor 1 temperature ls value
; TEMP2ms	    ; sensor 2 temperature ms value
; TEMP2ls		; sensor 2 temperature ls value

; processing. Rotate right three steps to have 0.5 degree resolution
; 0 bit is 0.5 degree resolution
; 9th bit, bit 8 is sign. (for processing, if set then clear, if clear then set) this gives a H000-1FF range,
; where 0 degrees is H100. 
; not actual temperatures are represented here since it is offset by D128 but it is easier to do comparisons. 
; 70 degrees for alarm is  B'1 0100 0110' or H146

PROCESS
; SENSOR1
	rrf		TEMP1ms,f
	rrf		TEMP1ls,f
	rrf		TEMP1ms,f
	rrf		TEMP1ls,f
	rrf		TEMP1ms,f
	rrf		TEMP1ls,f
; if bit 0 (9th bit) is set then clear. If clear then set.
	btfss	TEMP1ms,0
	goto	S1
	bcf		TEMP1ms,0
	goto	C1
S1	bsf		TEMP1ms,0
; clear bits 1-7 in ms byte
C1	movlw	B'0000001'
	andwf	TEMP1ms,f

; SENSOR2
	rrf		TEMP2ms,f
	rrf		TEMP2ls,f
	rrf		TEMP2ms,f
	rrf		TEMP2ls,f
	rrf		TEMP2ms,f
	rrf		TEMP2ls,f
; if bit 0 (9th bit) is set then clear. If clear then set.
	btfss	TEMP2ms,0
	goto	S2
	bcf		TEMP2ms,0
	goto	C2
S2	bsf		TEMP2ms,0
; clear bits 1-7 in ms byte
C2	movlw	B'0000001'
	andwf	TEMP2ms,f


;......................................................................................
; **** testing

; check timer status quo when watchdog timer reset
; write the two temperatures to flash for checking readings (plus others; see list below) 
; first 32 bytes H1F80-1F9F

	goto	ALM_FEATURE	; bypass testing with 'goto ALM_FEATURE'

; TEST to show temperature on RC6 PWM2 output
PWM_T_OUT

; If PORTB,7 low, show TEMP2;
	btfsc	PORTB,7
	goto	SHOW1

	movf	TEMP2ms,w
	movwf	TESTms
	movf	TEMP2ls,w
	movwf	TESTls
	goto	SHOW2

SHOW1
	movf	TEMP1ms,w
	movwf	TESTms
	movf	TEMP1ls,w
	movwf	TESTls
SHOW2
	rlf		TESTls,f
	rlf		TESTms,f
	rlf		TESTls,f
	rlf		TESTms,f
	rlf		TESTls,f
	rlf		TESTms,f
	rlf		TESTls,f
	rlf		TESTms,f
	rlf		TESTls,f
	rlf		TESTms,f
	rlf		TESTls,f
	rlf		TESTms,f
	rlf		TESTls,f
	rlf		TESTms,f
TR	movlb	D'12'
	movf	TESTms,w
	movwf	PWM2DCH	
	movf	TESTls,w
	andlw	B'10000000'
	movwf	PWM2DCL
	movlb	D'0'
; End test to PWM2

; Test store in Flash to see values
	goto	ALM_FEATURE		; bypass storing in flash

FLASH_STO ; store every 4s
	incf	RATE,f
	btfss	RATE,2
	goto	ALM_FEATURE
	clrf	RATE

; write the two temperatures to flash for checking readings (plus others; see list below) 
; first 32 bytes H1F80-1F9F
	movlw	H'1F'
	movwf	READADDMS		; MS Byte of temperature
	movlw	H'80'	
	movwf	READADDLS 		; LS Byte of temperature
; initial RAM location for TEMP1ms,ls and TEMP2ms,ls (H2013)
	movlw	H'20'
	movwf	FSR0H			; ms byte of address
	movlw	H'13'	
	movwf	FSR0L			; ls byte
	call	WRITE			; write to flash memory

; values stored in order
; TEMP1ms		; sensor 1 temperature ms value
; TEMP1ls		; sensor 1 temperature ls value
; TEMP2ms		; sensor 2 temperature ms value
; TEMP2ls		; sensor 2 temperature ls value
; BCD1			; setting for BCD switch BCD1
; BCD2			; setting for BCD switch BCD2
; BCD3			; setting for BCD switch BCD3
; BCD4			; setting for BCD switch BCD4
; MODE			; operating mode
; OVER			; temperature over flag
;......................................................................................

ALM_FEATURE

; ALARM FEATURE
; if JP1 link in, RB7 low

; check JP1 for alarm enabled
	btfsc	PORTB,7			; if set then bypass fire alarm
	goto	COMPARE

; if a sensor is disconnected then no fire alarm
	movf	SENSOR_OUT,w
	btfss	STATUS,Z
	goto	COMPARE			; sensor disconnected

; Fire Alarm selected and based on any sensor @>70 degrees = 140 (0.5 degrees/value)=(H18C)
; and\or rising >8 degrees C per minute. Set FIRE,0 when detected
; when on, loop to TEMP_LOOP for continuous check of temperature
; press S1 to silence siren, longer press Fire Alarm off


; compare sensors with 70 degrees C
	movf	TEMP1ms,w
	btfsc	STATUS,Z	; if not set then check ls byte 
	goto	T2OVR
	movf	TEMP1ls,w
	sublw	H'8C'		; over 70 degrees
	btfss	STATUS,C	; 
	goto	FIRE_ALM

T2OVR
	movf	TEMP2ms,w
	btfsc	STATUS,Z	; if not set then check ls byte 
	goto	DEG_RISE
	movf	TEMP2ls,w
	sublw	H'8C'		; over 70 degrees
	btfss	STATUS,C	; 
	goto	FIRE_ALM

DEG_RISE
; per minute
	btfss	MINUTE,1	; if set, first minute elapsed do checks
	goto	COMPARE

; check for degree rise/minute

	btfss	MINUTE,0	; if set, 1 minute elapsed
; otherwise 
	goto 	COMPARE

; minute flag, update and compare fire
	bcf		MINUTE,0	; flag cleared

; SENSOR1
; if old value is zero, then not updated yet so bypass comparison
	movf	TEMP1ms_OLD,w	; old temperature value for fire alarm detection ms byte
	btfss	STATUS,Z
	goto	CF1
	movf	TEMP1ls_OLD,w
	btfsc	STATUS,Z
	goto	UPDATE1
CF1
; compare new and old values
; if new value exceeds 8 degrees (16 x 0.5 degrees) compared to old value, set FIRE alarm 
; add 8 degrees (16) to old
	movlw	D'16'
	addwf	TEMP1ls_OLD,f	
; if carry then increase ms byte
	btfsc	STATUS,C
	incf	TEMP1ms_OLD,f
; compare
	movf	TEMP1ms_OLD,w
	subwf	TEMP1ms,w
	movwf	KEEP
	movf	TEMP1ls_OLD,w
	subwf	TEMP1ls,w
	btfss	STATUS,C
	decf	KEEP,f
	btfss	KEEP,7			; if clear then TEMP1ms/ls>=TEMP1ms/ls OLD
	goto	FIRE_ALM
; Update
UPDATE1
	movf	TEMP1ms,w
	movwf	TEMP1ms_OLD
	movf	TEMP1ls,w
	movwf	TEMP1ls_OLD

SENSOR2
; if old value is zero, then not updated yet so bypass comparison
	movf	TEMP2ms_OLD,w	; old temperature value for fire alarm detection ms byte
	btfss	STATUS,Z
	goto	CF2
	movf	TEMP2ls_OLD,w
	btfsc	STATUS,Z
	goto	UPDATE2
CF2
; compare new and old values
; if new value exceeds 8 degrees compared to old value, set FIRE alarm 
; add 8 degrees (16) to old
	movlw	D'16'
	addwf	TEMP2ls_OLD,f	
; if carry then increase ms byte
	btfsc	STATUS,C
	incf	TEMP2ms,f
; compare
	movf	TEMP2ms_OLD,w
	subwf	TEMP2ms,w
	movwf	KEEP
	movf	TEMP2ls_OLD,w
	subwf	TEMP2ls,w
	btfss	STATUS,C
	decf	KEEP,f
	btfss	KEEP,7			; if clear then TEMP2ms/ls>=TEMP2ms/ls OLD
	goto	FIRE_ALM
; Update
UPDATE2
	movf	TEMP2ms,w
	movwf	TEMP2ms_OLD
	movf	TEMP2ls,w
	movwf	TEMP2ls_OLD

NO_FIRE
; if no fire, then clear
	clrf	FIRE
	movlb	D'2'
	bcf		LATC,6			; relay2 off
	movlb	D'0'
	bsf		INTCON,GIE		; interrupts allowed
	goto	COMPARE
FIRE_ALM
; Set FIRE,0 when detected
; when on, loop to TEMP_LOOP for continuous check of temperature
; press S1 to silence siren, longer press Fire Alarm off
	bsf		FIRE,0			; alarm flag
	bcf		INTCON,GIE		; interrupts off
FIRE_2
; stop RELAY1
	movlb	D'2'
	bcf		LATC,3			; relay1 off
; sound alarm and flash LED, RELAY2 on
	bsf		LATC,6			; relay2 on
	movlb	D'0'	
; press S1 to silence buzzer
; buzzer works if FIRE,1 is clear
	movlb	D'2'
	btfss	FIRE,1
	bsf		LATC,7			; buzzer on
	bsf		LATB,6			; LED on
	movlb	D'0'
; delay
	call	DELAYms			; 100ms
	movlb	D'2'
	bcf		LATC,7			; buzzer off
	bcf		LATB,6			; LED off
	movlb	D'0'
	call	DELAYms			; 100ms

; if S1 pressed, silence buzzer
	btfsc	PORTB,5
	goto	FIRE_2		; 

	bsf		FIRE,1			; set so alarm siren is off
	call	DELAYms
	btfsc	PORTB,5
	goto	FIRE_2
	call	DELAYms
	btfsc	PORTB,5
	goto	FIRE_2
	call	DELAYms
	btfsc	PORTB,5
	goto	FIRE_2
	call	DELAYms
	btfsc	PORTB,5
	goto	FIRE_2
	call	DELAYms
	btfsc	PORTB,5
	goto	FIRE_2
; long switch press clear fire alarm for further testing
	clrf	FIRE
; stop RELAY1
	movlb	D'2'
	bcf		LATC,3			; relay1 off
; sound alarm and flash LED, RELAY2 off
	bcf		LATC,6			; relay2 off
	bcf		LATC,7			; buzzer off
	bcf		LATB,6			; LED off
	movlb	D'0'
	clrf	TEMP1ms_OLD
	clrf	TEMP1ls_OLD
	clrf	TEMP2ms_OLD
	clrf	TEMP2ls_OLD
OPE6
	call	DELAYms
	btfss	PORTB,5
	goto	OPE6			; wait for switch open
	goto	INIT			; exit fire alarm


COMPARE
; BCD1 has temperature difference setting Delta T
; 0=0.5 degrees, 1= 1degree, 2=1.5 degrees etc

; BCD2 has hysteresis
; 0=0 degrees, 1=0.5 degree, 2=1 degree, 3=1.5 degrees etc

; Comparisons
; check if fan running via OVER,1 and OVER,2 set
	btfsc	OVER,1		; if set then fan on heating
	goto	HEAT_TO_OFF	; test for switch off
	btfsc	OVER,2		; if set then fan is on cooling
	goto	COOL_TO_OFF	; test for switch off
; Fan off so test for requiring fan on

HEAT_TO_ON
; if T1 > T2 + Delta T then run fan, set OVER,0 Set OVER,1 for heat run
; add DELTA_T to T2. DELTA_T= 1 + BCD1
	movf	TEMP2ms,w	; ms byte of sensor 2
	movwf	CALCms		; temporary store
	movf	BCD1,w		; DELTA_T value which is BCD1
	addwf	TEMP2ls,w	; ls byte of sensor 2
	movwf	CALCls		; temporary store
	btfsc	STATUS,C	; if carry then increase ms byte
	incf	CALCms,f	; ms byte plus 1

; compare with T1
	movf	CALCms,w	; ms byte 
	subwf	TEMP1ms,w
	movwf	KEEP
	movf	CALCls,w
	subwf	TEMP1ls,w
	btfss	STATUS,C	; if carry clear, reduce ms byte
	decf	KEEP,f
	btfsc	KEEP,7		; if clear then T1 >= T2 + Delta T
	goto	COOL_TO_ON	; check for cool to on instead
; switch on fan
	bsf		OVER,0		; so fan switches on
	bsf		OVER,1		; so can test for when to switch off using heat to off test
	goto	MODE_T

COOL_TO_ON


; if T1 + Delta T < T2 then run fan, set OVER,0 and OVER,2 for cool run

; add DELTA_T to T1. DELTA_T= 1 + BCD1
	movf	TEMP1ms,w	; ms byte of sensor 1
	movwf	CALCms		; temporary store
	movf	BCD1,w		; DELTA_T value which is BCD1
	addwf	TEMP1ls,w	; ls byte of sensor 1
	movwf	CALCls		; temporary store
	btfsc	STATUS,C	; if carry then increase ms byte
	incf	CALCms,f	; ms byte plus 1

; compare with T2
	movf	TEMP2ms,w	; ms byte 
	subwf	CALCms,w
	movwf	KEEP
	movf	TEMP2ls,w
	subwf	CALCls,w
	btfss	STATUS,C	; if carry clear, reduce ms byte
	decf	KEEP,f
	btfss	KEEP,7		; if clear then T1 + Delta T =< T2
	goto	MODE_T
; switch on fan
	bsf		OVER,0		; so fan switches on
	bsf		OVER,2		; so can test for when to switch off using cool to off test
	goto	MODE_T

HEAT_TO_OFF

; if T1 < T2 + Delta T - Hysteresis then off, clear OVER for heat off

; subtract Hysteresis from delta T
	clrf	CALCms		; start with ms byte cleared
	movf	BCD2,w		; hysteresis = BCD2 
	subwf	BCD1,w		; delta T setting
	movwf	CALCls		; store
	btfss	STATUS,C	; if Hysteresis> delta T, clear CALCls ms byte
	clrf	CALCls

; add CALCms/ls to T2. 
	movf	TEMP2ms,w	; ms byte of sensor 2
	addwf	CALCms,f	; temporary store
	movf	CALCls,w	
	addwf	TEMP2ls,w	; ls byte of sensor 2
	movwf	CALCls		; temporary store
	btfsc	STATUS,C	; if carry then increase ms byte
	incf	CALCms,f	; ms byte plus 1

; compare with T1
	movf	CALCms,w	; ms byte 
	subwf	TEMP1ms,w
	movwf	KEEP
	movf	CALCls,w
	subwf	TEMP1ls,w
	btfss	STATUS,C	; if carry clear, reduce ms byte
	decf	KEEP,f
	btfss	KEEP,7		; if set then T1 < T2 + Delta T - Hysteresis
	goto	MODE_T		; 
; switch off fan
	clrf	OVER		; so fan switches off
	goto	MODE_T

COOL_TO_OFF

; if T1 + Delta T - hysteresis < T2 then off, clear OVER for cool off 

; subtract Hysteresis from delta T
	clrf	CALCms		; start with ms byte cleared
	movf	BCD2,w		; hysteresis = BCD2 
	subwf	BCD1,w		; delta T setting
	movwf	CALCls		; store
	btfss	STATUS,C	; if Hysteresis> delta T, clear CALCls ms byte
	clrf	CALCls

; add CALCms/ls to T1. DELTA_T= 1 + BCD1
	movf	TEMP1ms,w	; ms byte of sensor 1
	movwf	CALCms		; temporary store
	movf	BCD1,w		; DELTA_T value which is BCD1
	addwf	TEMP1ls,w	; ls byte of sensor 1
	movwf	CALCls		; temporary store
	btfsc	STATUS,C	; if carry then increase ms byte
	incf	CALCms,f	; ms byte plus 1

; compare with T2
	movf	TEMP2ms,w	; ms byte 
	subwf	CALCms,w
	movwf	KEEP
	movf	TEMP2ls,w
	subwf	CALCls,w
	btfss	STATUS,C	; if carry clear, reduce ms byte
	decf	KEEP,f
	btfsc	KEEP,7		; if clear then T1 + Delta T - hysteresis > T2
	goto	MODE_T
; switch off fan
	clrf	OVER		; fan off
	goto	MODE_T
	

;...................................................................................
; temperature mode
MODE_T

; for MODE0, MODE1 if temperature is over then flash LED if fan is off or alternate flash if fan is on
; for MODE3 only flash LED if fan is off but controller operation is on
	movf	MODE,w
	btfsc	STATUS,Z	; mode0
	goto	T_MODES_0_1
	movf	MODE,w
	xorlw	D'1'		; mode1
	btfss	STATUS,Z	; 
	goto	T_MODE2		; assumes MODE2 since MODE0 and MODE1 are tested

T_MODES_0_1
; if fan is on
	btfss	PORTC,3
	goto	FAN_OFF1
; on and delta T over light LED
	btfsc	OVER,0		; btfsc for momentary flash for delta T under threshold, btfss for momentary flash for delta T over threshold.
	goto	CLEAR_FLASH	; light LED
; otherwise 2Hz
	
	bsf		FLASH,0
	bcf		FLASH,1
	goto	LOOP

FAN_OFF1
	btfsc	OVER,0		; if over LED on
	goto	MOM_FLASH
	clrf	FLASH
; LED off
	movlb	D'2'		; bank 2
	bcf		LATB,6		; LED off
	movlb	D'0'
	goto	LOOP
MOM_FLASH
; momentary flash for when fan is off
	bcf		FLASH,0
	bsf		FLASH,1			; momentary flash
	goto	LOOP
CLEAR_FLASH
	clrf	FLASH
; LED on
	movlb	D'2'		; bank 2
	bsf		LATB,6		; LED on
	movlb	D'0'


T_MODE2
; test for MODE2
	movf	BCD4,w
	xorlw	D'2'			; if mode2 
	btfss	STATUS,Z
	goto	LOOP

; if auto is on
	btfss	AUTO_STATUS,0
	goto	AUTO_OFF1
; on and delta T over light LED and switch off fan
	btfsc	OVER,0
	goto	CLEAR_FLASH2	; light LED and fan on
; otherwise 2Hz
	
	bcf		FLASH,0
	bcf		FLASH,1
	movlb	D'2'		; bank 2
	bcf		LATC,3		; FAN off
	bcf		LATB,6		; LED off
	movlb	D'0'
	goto	LOOP

AUTO_OFF1

; momentary flash for when auto is off
	bcf		FLASH,0
	bsf		FLASH,1			; momentary flash
; LED off and fan off
	movlb	D'2'		; bank 2
	bcf		LATB,6		; LED off
	bcf		LATC,3		; fan off
	movlb	D'0'
	goto	LOOP

CLEAR_FLASH2
	clrf	FLASH
; LED on
	movlb	D'2'		; bank 2
	bsf		LATB,6		; LED on
	bsf		LATC,3		; fan on
	movlb	D'0'

	goto	LOOP

;/////////////////////////////////////////////////////////////

; delay loop 

DELAYxms
	movlw	D'3'		; 300ms
XMS						; x milli seconds (x sets time)
	movwf	TEMP_VAL
DLYLOOP
	call	DELAYms
	decfsz	TEMP_VAL,f
	goto	DLYLOOP

DELAYms ; 100ms/ cycle
;	clrwdt
	movlw	D'186'		; delay value
DELAYX
	movwf	STORE1		; STORE1 is number of loops value
LOOP8	
	movlw	H'B2'
DELDSP
	movwf	STORE2		; STORE2 is internal loop value	
LOOP9
	decfsz	STORE2,f
	goto	LOOP9
	decfsz	STORE1,f
	goto	LOOP8
	return

DELAY_INT ; for INTERRUPT use only!
; delay for during interrupt, 100ms
	movlw	D'186'		; delay value
DELAYI
	movwf	STOREI1		; STOREI1 is number of loops value
;	clrwdt				; was unchecked
LOOP1	
	movlw	H'B2'
	movwf	STOREI2		; STOREI2 is internal loop value	
LOOP2
	decfsz	STOREI2,f
	goto	LOOP2
	decfsz	STOREI1,f
	goto	LOOP1
	return
	
;.....................................................................
; temperature reading code DS18B20 thermometer

; sensor 1
; device reset
RESET1
	bcf		INTCON,GIE
; high impedance
	movlb	D'1'		; bank 1
	bsf		TRISC,2		; DQ, RC2 an input so pulled high
	movlb	D'0'		; bank 0

	clrf	PRESENCE1	; presence detect flag=0
; low output	
	bcf		PORTC,2
	movlb	D'1'		; bank 1
	bcf		TRISC,2		; an output
	movlb	D'0'		; bank 0

	movlw	D'100'		; 100 x 5us=500us
	call	WAIT
; high impedance
	movlb	D'1'		; bank 1
	bsf		TRISC,2		; DQ, RC2 an input so pulled high
	movlb	D'0'		; bank 0

	movlw	D'14'		; 14 x 5us=70us
	call	WAIT
; read presence detect pulse
	btfss	PORTC,2		; sensor 1 DQ pin
	incf	PRESENCE1,f	; set as 1 if low (ie sensor has driven line low)
	movlw	D'86'		; 86 x 5us=430us
	call	WAIT
	bsf		INTCON,GIE
	return

; sensor 1
; transmit byte
XMIT1
	movwf	DATA_BYTE	; data to send
	movlw	D'8'		; 8 bits of data
	movwf	COUNT
SEND_DATA1
	bcf		INTCON,GIE
; low output
	bcf		PORTC,2
	movlb	D'1'		; bank 1
	bcf		TRISC,2		; an output
	movlb	D'0'		; bank 0

	nop
	nop
	nop
	rrf		DATA_BYTE,f	; get first byte to carry bit
	movlb	D'1'		; bank 1
	btfsc	STATUS,C	; check carry
	bsf		TRISC,2		; RC2 an input so pulled high
	movlb	D'0'		; bank 0
	movlw	D'12'		; set wait period at 12 x 5us=60us
	call	WAIT
	
; RC2 an input if not already (depends on the carry bit previously)
; high impedance
	movlb	D'1'		; bank 1
	bsf		TRISC,2		; DQ, RC2 an input so pulled high
	movlb	D'0'		; bank 0

	nop
	nop
	decfsz	COUNT,f		; count down the number of bits sent
	goto	SEND_DATA1	; continue to send data for the 8-bits
	bsf		INTCON,GIE
	return				; all 8-bits sent

; sensor 1
; receive byte
RCV1
	movlw	D'8'		; 8 bits of data
	movwf	COUNT
	clrf	DATA_BYTE
RCV_DATA1
	bcf		INTCON,GIE
; low output
	bcf		PORTC,2
	movlb	D'1'		; bank 1
	bcf		TRISC,2		; an output
	movlb	D'0'		; bank 0

	nop
	nop
	nop
	nop
	nop
	nop
; high impedance
	movlb	D'1'		; bank 1
	bsf		TRISC,2		; DQ, RC2 an input so pulled high
	movlb	D'0'		; bank 0	

	nop
	nop
	nop
	nop
	movf	PORTC,w		; read DQ port
	andlw	B'00000100'	; mask out RC2
	addlw	D'255'		; carry is set if RC2 is set
	rrf		DATA_BYTE,f
	movlw	D'10'		; 10 x 5us=50us
	call	WAIT
	decfsz	COUNT,f		; count down bits received
	goto	RCV_DATA1	; continue receiving data
	bsf		INTCON,GIE
	return				; all 8 bits received
	
; sensor 2
; device reset
RESET2
	bcf		INTCON,GIE
; high impedance
	movlb	D'1'		; bank 1
	bsf		TRISB,4		; DQ, RB4 an input so pulled high
	movlb	D'0'		; bank 0 

	clrf	PRESENCE2	; presence detect flag=0
; low output
	movlb	D'2'		; bank 2
	bcf		LATB,4		; portb,4
	movlb	D'1'		; bank 1
	bcf		TRISB,4		; an output
	movlb	D'0'		; bank 0

	movlw	D'100'		; 100 x 5us=500us
	call	WAIT
; high impedance
	movlb	D'1'		; bank 1
	bsf		TRISB,4		; DQ, RB4 an input so pulled high
	movlb	D'0'		; bank 0

	movlw	D'14'		; 14 x 5us=70us
	call	WAIT
; read presence detect pulse
	btfss	PORTB,4		; sensor 2 DQ pin
	incf	PRESENCE2,f	; set as 1 if low (ie sensor has driven line low)
	movlw	D'86'		; 86 x 5us=430us
	call	WAIT
	bsf		INTCON,GIE
	return

; sensor 2
; transmit byte
XMIT2
	movwf	DATA_BYTE	; data to send
	movlw	D'8'		; 8 bits of data
	movwf	COUNT
SEND_DATA2
	bcf		INTCON,GIE
; low output
	movlb	D'2'		; bank 2
	bcf		LATB,4		; portb,4	
	movlb	D'1'		; bank 1
	bcf		TRISB,4		; an output
	movlb	D'0'		; bank 0

	nop
	nop
	nop
	rrf		DATA_BYTE,f	; get first byte to carry bit
	movlb	D'1'		; bank 1
	btfsc	STATUS,C	; check carry
	bsf		TRISB,4		; RB4 an input so pulled high
	movlb	D'0'		; bank 0
	movlw	D'12'		; set wait period at 12 x 5us=60us
	call	WAIT
; RB4 an input if not already (depends on the carry bit previously)
	movlb	D'1'		; bank 1
	bsf		TRISB,4		; DQ, RB4 an input so pulled high
	movlb	D'0'		; bank 0

	nop
	nop
	decfsz	COUNT,f		; count down the number of bits sent
	goto	SEND_DATA2	; continue to send data for the 8-bits
	bsf		INTCON,GIE
	return


; sensor 2
; receive byte
RCV2
	movlw	D'8'		; 8 bits of data
	movwf	COUNT
	clrf	DATA_BYTE
RCV_DATA2
	bcf		INTCON,GIE
; low output
	movlb	D'2'		; bank 2
	bcf		LATB,4		; portb,4
	movlb	D'1'		; bank 1
	bcf		TRISB,4		; an output
	movlb	D'0'		; bank 0

	nop
	nop
	nop
	nop
	nop
	nop
; high impedance
	movlb	D'1'		; bank 1
	bsf		TRISB,4		; DQ, RB4 an input so pulled high
	movlb	D'0'		; bank 0	

	nop
	nop
	nop
	nop
	movf	PORTB,w		; read DQ port
	andlw	B'00010000'	; mask out RB4
	addlw	D'255'		; carry is set if RB4 is set
	rrf		DATA_BYTE,f
	movlw	D'10'		; 10 x 5us=50us
	call	WAIT
	decfsz	COUNT,f		; count down bits received
	goto	RCV_DATA2	; continue receiving data
	bsf		INTCON,GIE
	return				; all 8 bits received
	

WAIT ; time delay

	movwf	TEMP		; 5us per value
WAIT_LOOP
	nop
	nop
	decfsz	TEMP,f
	goto	WAIT_LOOP
	return	


;........................................


; write to data memory
WRITE
	clrf	LOOPS			; counter
	movf	INTCON,w
	movwf	INTCON_STORE	; store GIE (bit7)
	bcf		INTCON,GIE 		; Disable interrupts so required sequences will execute properly
; load write latches
	movlb	D'03'			; bank 3
	movf	READADDMS,W		; Load upper 6 bits of address boundary			
	movwf	PMADRH 			;
	movf	READADDLS,W 	; Load lower 8 bits of erase address boundary
	movwf	PMADRL 			;

; erase first
	bcf		PMCON1,CFGS 	; Not configuration space
	bsf		PMCON1,FREE 	; Specify an erase operation
	bsf		PMCON1,WREN 	; Enable writes
	movlw	H'55'			; Start of required write sequence:
	movwf	PMCON2 			; Write 55h
	movlw	H'AA' 			;
	movwf	PMCON2 			; Write AAh
	bsf		PMCON1,WR 		; Set WR bit to begin erase
	nop					 	; NOP instructions are forced as processor writes all the program memory write latches simultaneously to program memory.
	nop	 
	bcf		PMCON1,WREN 	; Disable writes

; load latches
	movf	READADDMS,W		; Load upper 6 bits of  address boundary			
	movwf	PMADRH			;
	movf	READADDLS,W 	; Load lower 8 bits of erase address boundary
	movwf	PMADRL 			;

	bcf		PMCON1,CFGS		; Not configuration space
	bsf		PMCON1,WREN	 	; Enable writes
	bsf	    PMCON1,LWLO 	; Only Load Write Latches

LOOPW
	moviw	FSR0++ 			; Load first data byte into lower
	movwf	PMDATL 			;
	movlw	H'3F'	 		; Load second data byte into upper
	movwf	PMDATH 			;

	movf 	LOOPS,w		 	; Check lower bits of address
	xorlw	0x9				; Check if we're on the last address
	andlw	0x1F ;
	btfsc	STATUS,Z 		; Exit if last of 32 words,
	goto	START_WRITE 	;

	movlw	H'55'		 	; Start of required write sequence:
	movwf	PMCON2 			; Write 55h
	movlw	H'AA' ;
	movwf	PMCON2			; Write AAh
	bsf		PMCON1,WR 		; Set WR bit to begin write
	nop					 	; nop instructions are forced as processor loads program memory write latches
	nop ;
	incf	PMADRL,f	 	; Still loading latches Increment address
	incf	LOOPS,f
	goto	LOOPW

START_WRITE ; write
	bcf		PMCON1,LWLO		; No more loading latches - Start Flash program
; memory write
	movlw	H'55'			; Start of required write sequence:
	movwf	PMCON2 			; Write 55h
	movlw	H'AA' ;
	movwf	PMCON2 			; Write AAh
	bsf		PMCON1,WR 		; Set WR bit to begin write
	nop						; NOP instructions are forced as processor writes all the program memory write latches simultaneously to program memory.
	nop	  
	bcf		PMCON1,WREN 	; Disable writes
	movlb	D'0'

	btfsc	INTCON_STORE,7	; if set then set GIE
	bsf		INTCON,GIE		; re-enable interrupts 
	return

; 	

 end